// ===============================
// ☕ Coffee Log (JSON) app.js
// ===============================

document.addEventListener("DOMContentLoaded", () => {

  // --- タブ切り替え ---
  const tabs = document.querySelectorAll(".tab");
  const sections = document.querySelectorAll(".panes section");

  tabs.forEach(tab => {
    tab.addEventListener("click", () => {
      tabs.forEach(t => t.classList.remove("active"));
      sections.forEach(s => s.classList.remove("active"));
      tab.classList.add("active");
      document.getElementById(tab.dataset.target).classList.add("active");
    });
  });

  // --- 共通：安全なカード生成関数 ---
  function createCard(item, title = null) {
    const div = document.createElement("div");
    div.className = "card";

    // 内部構造を配列で定義してループで回すと保守性が上がります
    const contents = [
      { label: "", value: title || item.date, isDate: true },
      { label: "☕ 豆：", value: item.bean },
      { label: "🔥 焙煎：", value: item.roast },
      { label: "🩸 抽出：", value: item.method },
      { label: "🏠 店：", value: item.shop || "-" },
      { label: "💬 ", value: item.comment || "" }
    ];

    contents.forEach(row => {
      const rowDiv = document.createElement("div");
      if (row.isDate) rowDiv.className = "date";
      // textContentを使うことで、タグが含まれていても安全に表示されます
      rowDiv.textContent = `${row.label}${row.value}`;
      div.appendChild(rowDiv);
    });

    return div;
  }

  // --- 保存処理 ---
  const saveBtn = document.getElementById("saveBtn");
  const saveMessage = document.getElementById("saveMessage");
  const lastSaved = document.getElementById("lastSaved");

  saveBtn.addEventListener("click", () => {
    const fields = ["bean", "roast", "method", "shop", "comment"];
    const data = {};
    fields.forEach(f => data[f] = document.getElementById(f).value.trim());

    if (!data.bean || !data.method) {
      saveMessage.textContent = "⚠ 豆と抽出方法は必須です";
      saveMessage.style.color = "red";
      return;
    }

// --- 追加：文字数バリデーション ---
    if (data.bean.length > 20 || data.roast.length > 20 || 
        data.method.length > 20 || data.shop.length > 20) {
      saveMessage.textContent = "⚠ 各項目（豆・焙煎・抽出・店）は20文字以内で入力してください";
      saveMessage.style.color = "red";
      return;
    }

    if (data.comment.length > 30) {
      saveMessage.textContent = "⚠ コメントは30文字以内で入力してください";
      saveMessage.style.color = "red";
      return;
    }
    // --- 追加ここまで ---

    fetch("api/save.php", {
      method: "POST",
      headers: { "Content-Type": "application/json" },
      body: JSON.stringify(data)
    })
    .then(res => {
      if (!res.ok) throw new Error();
      return res.text();
    })
    .then(() => {
      saveMessage.textContent = "☕ 記録しました";
      saveMessage.style.color = "green";

      // 保存内容の確認表示（安全な関数を使用）
      lastSaved.innerHTML = "";
      lastSaved.appendChild(createCard(data, "保存内容の確認"));

      // 入力欄クリア
      fields.forEach(f => document.getElementById(f).value = "");
    })
    .catch(() => {
      saveMessage.textContent = "⚠ 保存に失敗しました";
      saveMessage.style.color = "red";
    });
  });

  // --- 検索処理 ---
  const searchBtn = document.getElementById("searchBtn");
  const resultsDiv = document.getElementById("results");

  searchBtn.addEventListener("click", () => {
    const type = document.getElementById("type").value;
    const keyword = document.getElementById("keyword").value.trim();

    fetch("api/list.php", { cache: "no-store" })
      .then(res => res.json())
      .then(data => {
        const filtered = (keyword !== "" && type !== "all")
          ? data.filter(item => (item[type] ?? "").includes(keyword))
          : data;
        
        renderResults(filtered);
      });
  });

  function renderResults(list) {
    resultsDiv.innerHTML = "";
    if (list.length === 0) {
      resultsDiv.innerHTML = "<p>該当する記録はありません。</p>";
      return;
    }
    list.forEach(item => {
      resultsDiv.appendChild(createCard(item));
    });
  }
});